// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "gp_internal.h"
#include "gp_draw_internal.h"
#include "main.h"

template<class T> void GP::draw_primitive() {
	BYTE vat = queue_get_past_byte(1) & 0x07;
	WORD numvertices = GP_QUEUE_GET_WORD;
	if(numvertices == 0) {
		GPDEGUB("GP Draw %s 0\n", T::name);
		stat.odraw_null++;
		return;
	}
	if(T::VertexCountIsBad(numvertices)) {
		DEGUB("%i vertices!\n", numvertices);
		throw hardware_fatal_exception("GP Draw "+ string(T::name) +" bad vertex count!");
	}

	CONVERTICES c;
	getConvertices(vat, numvertices, T::name, c);
	do_stuff_before_draw(c.FVF, c.midx);

	WORD primitiveCount = T::PrimitiveCount(numvertices);
	T::StatPrimitives(stat) += primitiveCount;
#pragma warning(push)
#pragma warning(disable:4127)
	if(!((vs_control.cull_all && T::isSurface) || vs_control.lo_noop)) {
#pragma warning(pop)
		GPHR(m.pd3dDevice->SetStreamSource(0, c.vertexBuffer, 0, c.convertexSize));
		T::Draw(this, primitiveCount);
	}
	T::StatOps(stat)++;

	c.vertexBuffer->Release();
}

const char* const DrawQuads::name = "Quads";
LPDIRECT3DINDEXBUFFER9 DrawQuads::mIndices = NULL;
DWORD DrawQuads::mNIndices = 0;
void GP::draw_quads() {
	draw_primitive<DrawQuads>();
}

const char* const DrawTriangles::name = "Triangles";
void GP::draw_triangles() {
	draw_primitive<DrawTriangles>();
}

const char* const DrawTriangleStrip::name = "Triangle strip";
void GP::draw_triangle_strip() {
	draw_primitive<DrawTriangleStrip>();
}

const char* const DrawTriangleFan::name = "Triangle fan";
void GP::draw_triangle_fan() {
	draw_primitive<DrawTriangleFan>();
}

const char* const DrawLines::name = "Lines";
void GP::draw_lines() {
	draw_primitive<DrawLines>();
}

const char* const DrawLineStrip::name = "Line strip";
void GP::draw_line_strip() {
	draw_primitive<DrawLineStrip>();
}

const char* const DrawPoints::name = "Points";
void GP::draw_points() {
	draw_primitive<DrawPoints>();
}

void GP::getConvertices(BYTE vat, WORD numvertices, const char* primstr,
												CONVERTICES& results)
{
	GPDEGUB("GP Draw %s, %i vertices, VAT %i:\n", primstr, numvertices, vat);
	MYASSERT(vat <= 0x07);
	int vertex_size=0;

	//Read vertex format bitfields

	if((m.cp_reg[0x60] & 0xFFFF0000) != 0)
		throw hardware_fatal_exception("GP VCD: Unused bits set!");
	if(!getbitr(m.cp_reg[0x80 + vat], 31))
		throw hardware_fatal_exception("GP VCD: VCACHE_ENHANCE not set!");

	WORD midx = WORD(m.cp_reg[0x50] & 0x1FF);
	if(midx & 0x1FC)
		throw hardware_fatal_exception("GP VCD: Unemulated Texture Matrix Indices");

	int pos_enc = getbitsr(m.cp_reg[0x50], 10, 9);
	bool pos_3d = getbitr(m.cp_reg[0x70 + vat], 0);
	int pos_fmt = getbitsr(m.cp_reg[0x70 + vat], 3, 1);
	int pos_shift = getbitsr(m.cp_reg[0x70 + vat], 8, 4);

	int norm_enc = getbitsr(m.cp_reg[0x50], 12, 11);
	bool norm_nine = getbitr(m.cp_reg[0x70 + vat], 9);
	int norm_fmt = getbitsr(m.cp_reg[0x70 + vat], 12, 10);

	int col0_enc = getbitsr(m.cp_reg[0x50], 14, 13);
	bool col0_alpha = getbitr(m.cp_reg[0x70 + vat], 13);
	int col0_fmt = getbitsr(m.cp_reg[0x70 + vat], 16, 14);

	int col1_enc = getbitsr(m.cp_reg[0x50], 16, 15);
	bool col1_alpha = getbitr(m.cp_reg[0x70 + vat], 17);
	int col1_fmt = getbitsw(m.cp_reg[0x70 + vat], 20, 18);

	TEX_VCD tex[8];
	for(int i=0; i<8; i++) {
		tex[i].enc = getbitsr(m.cp_reg[0x60], i*2+1, i*2);

		tex[i].fmt = getbitsr(m.cp_reg[tvib_two_fmt[i].reg + vat],
			tvib_two_fmt[i].start + 3, tvib_two_fmt[i].start + 1);
		tex[i].cnt = getbitr(m.cp_reg[tvib_two_fmt[i].reg + vat], tvib_two_fmt[i].start);
		tex[i].shift = getbitsr(m.cp_reg[tvib_shift[i].reg + vat],
			tvib_shift[i].start + 4, tvib_shift[i].start);
	}

	//Calculate size of queue vertex
	for(int i=0; i<9; i++) {
		if(getbitr(midx, i)) {
			vertex_size += 1;
		}
	}
	switch(pos_enc) {
	case 1:
		vertex_size += (pos_3d ? 3 : 2) * gp_get_coord_size(pos_fmt);
		break;
	case 2:
		vertex_size += 1;
		break;
	case 3:
		vertex_size += 2;
		break;
	}
	switch(norm_enc) {
	case 0:
		break;
	case 1:
		vertex_size += (norm_nine ? 9 : 3) * gp_get_norm_size(norm_fmt);
		break;
	case 2:
		vertex_size += 1;
		break;
	case 3:
		vertex_size += 2;
		break;
	}
	switch(col0_enc) {
	case 0:
		break;
	case 1:
		vertex_size += gp_get_color_size(col0_fmt);
		break;
	case 2:
		vertex_size += 1;
		break;
	case 3:
		vertex_size += 2;
		break;
	}
	switch(col1_enc) {
	case 0:
		break;
	case 1:
		vertex_size += gp_get_color_size(col1_fmt);
		break;
	case 2:
		vertex_size += 1;
		break;
	case 3:
		vertex_size += 2;
		break;
	}
	for(int i=0; i<8; i++) {
		switch(tex[i].enc) {
		case 0:
			break;
		case 1:
			vertex_size += (tex[i].cnt ? 2 : 1) * gp_get_coord_size(tex[i].fmt);
			break;
		case 2:
			vertex_size += 1;
			break;
		case 3:
			vertex_size += 2;
			break;
		}
	}

	//Set up vertex format
	DWORD FVF = 0;
	BYTE convertex_size = 0;

	FVF = D3DFVF_XYZ;
	convertex_size = 4*3;
	GPDEGUB("Position: %s, type=%s, fmt=%s, shift=%i\n",
		gp_get_enc_string(pos_enc), pos_3d ? "3D" : "2D",
		gp_get_coord_size_string(pos_fmt), pos_shift);

	{
		int count = 0;
		bool error = false;
		GPDEGUB("Matrix indices:");
		if(midx) for(int i=0; i<9; i++) {
			if(getbitr(midx, i)) {
				if(i != count)
					error = true;
				if(i==0) {
					GPDEGUB(" PN");
				} else {
					GPDEGUB("%s TEX%i", count == 0 ? "" : ",", i);
				}
				count++;
			}
		}
		if(count == 0) {
			GPDEGUB(" not present");
		} else {
			FVF = D3DFVF_XYZB1 | D3DFVF_LASTBETA_UBYTE4;
			convertex_size = 4*4;
		}
		GPDEGUB("\n");
		if(error)
			throw hardware_fatal_exception("GP Unemulated Matrix Indices!");
		if(count > 2)
			throw hardware_fatal_exception("More than 2 GP Texture matrix indices not "
			"supported!");
	}

	GPDEGUB("Normal: %s", gp_get_enc_string(norm_enc));
	if(norm_enc) {
		GPDEGUB(", count=%s, fmt=%s", norm_nine ? "nine" : "three",
			gp_get_norm_size_string(norm_fmt));
		if(norm_nine)
			throw hardware_fatal_exception("GP Binormals not supported!");
		FVF |= D3DFVF_NORMAL;
		convertex_size += 4*3;
	}
	GPDEGUB("\n");

	GPDEGUB("Color 0: %s", gp_get_enc_string(col0_enc));
	if(col0_enc) {
		GPDEGUB(", type=%s%s", col0_alpha ? "RGBA" : "RGB",
			gp_get_color_string(col0_fmt));
		FVF |= D3DFVF_DIFFUSE;
		convertex_size += 4;
		if(col0_alpha ? (col0_fmt < 3) : (col0_fmt > 2))
			throw hardware_fatal_exception("GP Diffuse Color invalid type!");
	}
	GPDEGUB("\n");

	GPDEGUB("Color 1: %s", gp_get_enc_string(col1_enc));
	if(col1_enc) {
		GPDEGUB(", type=%s, fmt=%s", col1_alpha ? "RGBA" : "RGB",
			gp_get_color_string(col1_fmt));
		FVF |= D3DFVF_SPECULAR;
		convertex_size += 4;
		if(col1_alpha ? (col1_fmt < 3) : (col1_fmt > 2))
			throw hardware_fatal_exception("GP Specular Color invalid type!");
	}
	GPDEGUB("\n");

	{
		bool b4 = true;
		int ntex = 0;
		for(int i=0; i<8; i++) {
			GPDEGUB("Texture %i: %s", i, gp_get_enc_string(tex[i].enc));
			if(tex[i].enc) {
				GPDEGUB(", type=%s, fmt=%s, shift=%i", tex[i].cnt ? "2D" : "1D",
					gp_get_coord_size_string(tex[i].fmt), tex[i].shift);
				FVF |= (tex[i].cnt ? D3DFVF_TEXCOORDSIZE2(i) : D3DFVF_TEXCOORDSIZE1(i));
				convertex_size += tex[i].cnt ? 4*2 : 4*1;
				if(!b4)
					throw hardware_fatal_exception("GP vertex format gap!");
				ntex = i+1;
			} else
				b4 = false;
			GPDEGUB("\n");
		}
		FVF |= ntex << D3DFVF_TEXCOUNT_SHIFT;
	}

	GPDEGUB("Total vertex size: %i bytes\n", vertex_size);
	GPDEGUB("Convertex size: %i bytes\n", convertex_size);

	//Read indexed arrays (if need be)
#define ARRAY_POS_BASE PHYSICALIZE(m.cp_reg[0xA0])
#define ARRAY_POS_STRIDE (m.cp_reg[0xB0])
#define ARRAY_NORM_BASE PHYSICALIZE(m.cp_reg[0xA1])
#define ARRAY_NORM_STRIDE (m.cp_reg[0xB1])
#define ARRAY_COL0_BASE PHYSICALIZE(m.cp_reg[0xA2])
#define ARRAY_COL0_STRIDE (m.cp_reg[0xB2])
#define ARRAY_COL1_BASE PHYSICALIZE(m.cp_reg[0xA3])
#define ARRAY_COL1_STRIDE (m.cp_reg[0xB3])
#define ARRAY_TEX_BASE(i) PHYSICALIZE(m.cp_reg[0xA4 + i])
#define ARRAY_TEX_STRIDE(i) (m.cp_reg[0xB4 + i])

	//D3D doesn't support individual vertex components at all,
	//thus we must make the converted list fully direct.
	//there is a possibility that it could be made indexed,
	//but it would require some vertices to be identical.

	LPDIRECT3DVERTEXBUFFER9 vertexBuffer;
	GPHR(m.pd3dDevice->CreateVertexBuffer(convertex_size * numvertices,
		D3DUSAGE_WRITEONLY, FVF, D3DPOOL_DEFAULT, &vertexBuffer, NULL));
	BYTE* vertexPointer;
	GPHR(vertexBuffer->Lock(0, 0, (void**)&vertexPointer, D3DLOCK_DISCARD));

#define CV_ADD_FLOAT(f) *MAKEP(float, vertexPointer) = f; vertexPointer += 4
#define CV_ADD_DWORD(d) *MAKEP(DWORD, vertexPointer) = d; vertexPointer += 4

	//I think these addresses may be physical and require no translation
	BYTE *array_pos=NULL;
	if(pos_enc == 2 || pos_enc == 3)
		array_pos = mem.getp_physical(ARRAY_POS_BASE, 0x10000); //tentative maximum sizes. fix.
	BYTE *array_norm=NULL;
	if(norm_enc == 2 || norm_enc == 3)
		array_norm = mem.getp_physical(ARRAY_NORM_BASE, 0x10000);
	BYTE *array_col0=NULL;
	if(col0_enc == 2 || col0_enc == 3)
		array_col0 = mem.getp_physical(ARRAY_COL0_BASE, 0x10000);
	BYTE *array_col1=NULL;
	if(col1_enc == 2 || col1_enc == 3)
		array_col1 = mem.getp_physical(ARRAY_COL1_BASE, 0x10000);
	BYTE *array_tex[8]={NULL};
	for(int i=0; i<8; i++) {
		if(tex[i].enc == 2 || tex[i].enc == 3)
			array_tex[i] = mem.getp_physical(ARRAY_TEX_BASE(i), 0x10000);
	}

	//gp_queue_pos += 3;
	for(int j=0; j<numvertices; j++) {

		DWORD matrix_index = 0;
		if(midx) {
			BYTE *indices = (BYTE*)&matrix_index;
			int count = 0;//2;
			GPDEGUB("Matrix index:");
			for(int i=0; i<9; i++) {
				if(getbitr(midx, i)) {
					BYTE index = GP_QUEUE_GET_BYTE;
					GPDEGUB(" %i", index);
					if(index % 3 != 0)
						throw hardware_fatal_exception("GP Unemulated Vertex Matrix Index!");
					index /= 3;
					if(i == 0) {	//pos/nrm
						if(index > 9)
							throw hardware_fatal_exception("GP Unemulated Vertex Matrix Index!");
						//indices[0] = index*4;  //should become v.x
						//indices[1] = index*3;  //v.y
						indices[count++] = index;
					} else {  //tex
						MYASSERT(count < 4);
						if(index < 10 || index > 20)
							throw hardware_fatal_exception("GP Unemulated Vertex Matrix Index!");
						index -= 10;
						//indices[count++] = index*3;	//v.z and z.w
						indices[count++] = index;
					}
				}
			}
			GPDEGUB(" ");
		}

		GPDEGUB("Pos:");
		const BYTE *pointer;
		switch(pos_enc) {
	case 1:
		pointer = (BYTE*)queue_get_data((pos_3d ? 3 : 2) * gp_get_coord_size(pos_fmt));
		break;
	case 2:
		{
			BYTE index = GP_QUEUE_GET_BYTE;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_pos[index * ARRAY_POS_STRIDE];
		}
		break;
	case 3:
		{
			WORD index = GP_QUEUE_GET_WORD;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_pos[index * ARRAY_POS_STRIDE];
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Pos Encoding!");
		}
		for(int k=0; k<(pos_3d?3:2); k++) {
			switch(pos_fmt) {
	case 0:
		{
			BYTE b = *MAKEP(BYTE, pointer);
			float f = float(b) / (1 << pos_shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, b);
			pointer += 1;
		}
		break;
	case 1:
		{
			char c = *MAKEP(char, pointer);
			float f = float(c) / (1 << pos_shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, c);
			pointer += 1;
		}
		break;
	case 2:
		{
			WORD w = swaph(*MAKEP(WORD, pointer));
			float f = float(w) / (1 << pos_shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, w);
			pointer += 2;
		}
		break;
	case 3:
		{
			short s = swaph(*MAKEP(short, pointer));
			float f = float(s) / (1 << pos_shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, s);
			pointer += 2;
		}
		break;
	case 4:
		{
			DWORD d = swapw(*MAKEP(DWORD, pointer));
			float f = MAKE(float, d);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%08X)", f, d);
			pointer += 4;
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Pos Size!");
			}
		}
		if(!pos_3d) {
			CV_ADD_FLOAT(0);
		}

		if(midx) {
			CV_ADD_DWORD(matrix_index);
		}

		if(norm_enc != 0) {
			GPDEGUB(" Norm:");
			const BYTE *pointer;
			switch(norm_enc) {
	case 1:
		pointer = (BYTE*)queue_get_data((norm_nine ? 9 : 3) *
			gp_get_coord_size(norm_fmt));
		break;
	case 2:
		{
			BYTE index = GP_QUEUE_GET_BYTE;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_norm[index * ARRAY_NORM_STRIDE];
		}
		break;
	case 3:
		{
			WORD index = GP_QUEUE_GET_WORD;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_norm[index * ARRAY_NORM_STRIDE];
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Tex0 Encoding!");
			}
			for(int k=0; k<(norm_nine ? 9 : 3); k++) {
				switch(norm_fmt) {
	case 0:
		{
			BYTE b = *MAKEP(BYTE, pointer);
			float f = float(b) / (1 << 6);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%02X)", f, b);
			pointer += 1;
		}
		break;
	case 1:
		{
			char c = *MAKEP(char, pointer);
			float f = float(c) / (1 << 6);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, c);
			pointer += 1;
		}
		break;
	case 2:
		{
			WORD w = swaph(*MAKEP(WORD, pointer));
			float f = float(w) / (1 << 14);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%04X)", f, w);
			pointer += 2;
		}
		break;
	case 3:
		{
			short s = swaph(*MAKEP(short, pointer));
			float f = float(s) / (1 << 14);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, s);
			pointer += 2;
		}
		break;
	case 4:
		{
			DWORD d = swapw(*MAKEP(DWORD, pointer));
			float f = MAKE(float, d);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%08X)", f, d);
			pointer += 4;
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Normal Size!");
				}
			}
		}

		if(col0_enc != 0) {
			GPDEGUB(" Col0: ");
			const BYTE *pointer;
			switch(col0_enc) {
	case 1:
		pointer = (BYTE*)queue_get_data(gp_get_color_size(col0_fmt));
		break;
	case 2:
		{
			BYTE index = GP_QUEUE_GET_BYTE;  //are these indices?
			GPDEGUB("(%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_col0[index * ARRAY_COL0_STRIDE];
		}
		break;
	case 3:
		{
			WORD index = GP_QUEUE_GET_WORD;
			GPDEGUB("(%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_col0[index * ARRAY_COL0_STRIDE];
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad color0 Encoding!");
			}
			switch(gp_get_color_size(col0_fmt)) {  //note: switch on data size, not data type!
	case 2:
		GPDEGUB("%04X", swaph(*MAKEP(WORD, pointer)));  //clean up
		break;
	case 3:
		GPDEGUB("%06X", make_24bits(MAKEP(BYTE, pointer)));
		break;
	case 4:
		GPDEGUB("%08X", swapw(*MAKEP(DWORD, pointer))); //clean up
		break;
	default:
		MYASSERT(false);
			}
			DWORD d;
			switch(col0_fmt) {
	case 0:	//rgb565
		d = gp_argb8_rgb565(swaph(*MAKEP(WORD, pointer)));
		break;
	case 1: //rgb8
		d = make_24bits(MAKEP(BYTE, pointer)) | 0xFF000000;
		break;
	case 2: //rgbx8
		d = _rotr(swapw(*MAKEP(DWORD, pointer)), 8) | 0xFF000000;
		break;
	case 3: //rgba4
		d = gp_argb8_rgba4(swaph(*MAKEP(WORD, pointer)));
		break;
	case 4: //rgba6
		d = gp_argb8_rgba6(make_24bits(MAKEP(BYTE, pointer)));
		break;
	case 5: //rgba8
		d = _rotr(swapw(*MAKEP(DWORD, pointer)), 8);
		break;
	default:
		throw hardware_fatal_exception("GP unemulated color format!");
			}
			GPDEGUB("(%08X)", d);
			CV_ADD_DWORD(d);
		}

		if(col1_enc != 0) {
			GPDEGUB(" Col0: ");
			const BYTE *pointer;
			switch(col1_enc) {
	case 1:
		pointer = (BYTE*)queue_get_data(gp_get_color_size(col1_fmt));
		break;
	case 2:
		{
			BYTE index = GP_QUEUE_GET_BYTE;  //are these indices?
			GPDEGUB("(%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_col1[index * ARRAY_COL0_STRIDE];
		}
		break;
	case 3:
		{
			WORD index = GP_QUEUE_GET_WORD;
			GPDEGUB("(%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &array_col1[index * ARRAY_COL0_STRIDE];
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad color1 Encoding!");
			}
			switch(gp_get_color_size(col1_fmt)) {  //note: switch on data size, not data type!
	case 2:
		GPDEGUB("%04X", swaph(*MAKEP(WORD, pointer)));  //clean up
		break;
	case 3:
		GPDEGUB("%06X", make_24bits(MAKEP(BYTE, pointer)));
		break;
	case 4:
		GPDEGUB("%08X", swapw(*MAKEP(DWORD, pointer))); //clean up
		break;
	default:
		MYASSERT(false);
			}
			DWORD d;
			switch(col1_fmt) {
	case 0:	//rgb565
		d = gp_argb8_rgb565(swaph(*MAKEP(WORD, pointer)));
		break;
	case 1: //rgb8
		d = make_24bits(MAKEP(BYTE, pointer)) | 0xFF000000;
		break;
	case 2: //rgbx8
		d = _rotr(swapw(*MAKEP(DWORD, pointer)), 8) | 0xFF000000;
		break;
	case 3: //rgba4
		d = gp_argb8_rgba4(swaph(*MAKEP(WORD, pointer)));
		break;
	case 4: //rgba6
		d = gp_argb8_rgba6(make_24bits(MAKEP(BYTE, pointer)));
		break;
	case 5: //rgba8
		d = _rotr(swapw(*MAKEP(DWORD, pointer)), 8);
		break;
	default:
		throw hardware_fatal_exception("GP unemulated color format!");
			}
			GPDEGUB("(%08X)", d);
			CV_ADD_DWORD(d);
		}

		for(int i=0; i<8; i++) if(tex[i].enc != 0) {
			GPDEGUB(" Tex0:");
			const BYTE *pointer;
			switch(tex[i].enc) {
	case 1:
		pointer = (BYTE*)queue_get_data((tex[i].cnt ? 2 : 1) *
			gp_get_coord_size(tex[i].fmt));
		break;
	case 2:
		{
			BYTE index = GP_QUEUE_GET_BYTE;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &(array_tex[i])[index * ARRAY_TEX_STRIDE(i)];
		}
		break;
	case 3:
		{
			WORD index = GP_QUEUE_GET_WORD;
			GPDEGUB(" (%u)", index);
			if(index == -1)
				throw hardware_fatal_exception("GP Indexed Vertex Disable unemulated!");
			pointer = &(array_tex[i])[index * ARRAY_TEX_STRIDE(i)];
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Tex0 Encoding!");
			}
			for(int k=0; k<(tex[i].cnt ? 2 : 1); k++) {
				switch(tex[i].fmt) {
	case 0:
		{
			BYTE b = *MAKEP(BYTE, pointer);
			float f = float(b) / (1 << tex[i].shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%02X)", f, b);
			pointer += 1;
		}
		break;
	case 1:
		{
			char c = *MAKEP(char, pointer);
			float f = float(c) / (1 << tex[i].shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, c);
			pointer += 1;
		}
		break;
	case 2:
		{
			WORD w = swaph(*MAKEP(WORD, pointer));
			float f = float(w) / (1 << tex[i].shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%04X)", f, w);
			pointer += 2;
		}
		break;
	case 3:
		{
			short s = swaph(*MAKEP(short, pointer));
			float f = float(s) / (1 << tex[i].shift);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(%i)", f, s);
			pointer += 2;
		}
		break;
	case 4:
		{
			DWORD d = swapw(*MAKEP(DWORD, pointer));
			float f = MAKE(float, d);
			CV_ADD_FLOAT(f);
			GPDEGUB(" %.6g(0x%08X)", f, d);
			pointer += 4;
		}
		break;
	default:
		throw hardware_fatal_exception("GP Bad Tex0 Size!");
				}
			}
		}
		GPDEGUB("\n");
	}

	GPHR(vertexBuffer->Unlock());
	results.vertexBuffer = vertexBuffer;
	results.FVF = FVF;
	results.midx = midx;
	results.convertexSize = convertex_size;
}

void GP::do_stuff_before_draw(DWORD FVF, WORD midx) {
	setVertexShader(FVF, midx);
	setPixelProcessing();

	for(BYTE i=0; i<8; i++) {
		if(tx[i].changed) {
			do_texture(i);
		}
	}
	if(m.scissor.changed) {
		GPHR(setScissorTest());
	}
	if(!m.in_scene) {
		GPHR(beginScene());
	}
	if(vs_key.ncol != ps_key.ncol)
		throw hardware_fatal_exception("GP ncol discrepancy!");
	if(vs_key.ntex != ps_key.ntex)
		throw hardware_fatal_exception("GP ntex discrepancy!");
}
HRESULT GP::setScissorTest() {
	RECT rect = { m.scissor.left + m.scissor.xoffset, m.scissor.top + m.scissor.yoffset,
		m.scissor.right + m.scissor.xoffset, m.scissor.bottom + m.scissor.yoffset };
	THR(setRS(D3DRS_SCISSORTESTENABLE, true));
	CLAMPLOW(rect.left, 0);
	CLAMPLOW(rect.top, 0);
	CLAMPHIGH(rect.right, WIDTH);
	CLAMPHIGH(rect.bottom, HEIGHT);
	THR(m.pd3dDevice->SetScissorRect(&rect));
	m.scissor.changed = false;
	VGPDEGUB("Scissorbox set: %i, %i, %i, %i\n", rect.left, rect.top, rect.right,
		rect.bottom);
	return S_OK;
}

const char *gp_get_coord_size_string(int i) {
	static char buffer[32];
	switch(i) {
	case 0:
		return "ubyte";
	case 1:
		return "byte";
	case 2:
		return "ushort";
	case 3:
		return "short";
	case 4:
		return "float";
	default:
		//sprintf(buffer, "Reserved (%i)", i);
		//return buffer;
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Reserved Coord size!");
	}
}
int gp_get_coord_size(int i) {
	switch(i) {
	case 0:
		return 1;
	case 1:
		return 1;
	case 2:
		return 2;
	case 3:
		return 2;
	case 4:
		return 4;
	default:
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Coord Reserved size!");
	}
}

const char *gp_get_norm_size_string(int i) {
	static char buffer[32];
	switch(i) {
	case 1:
		return "byte";
	case 3:
		return "short";
	case 4:
		return "float";
	default:
		//sprintf(buffer, "Reserved (%i)", i);
		//return buffer;
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Normal Reserved size!");
	}
}
int gp_get_norm_size(int i) {
	switch(i) {
	case 1:
		return 1;
	case 3:
		return 2;
	case 4:
		return 4;
	default:
		//return 0;
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Normal Reserved size!");
	}
}

const char *gp_get_color_string(int i) {
	static char buffer[32];
	switch(i) {
	case 0:
		return "565";
	case 1:
		return "888";
	case 2:
		return "888x";
	case 3:
		return "4444";
	case 4:
		return "6666";
	case 5:
		return "8888";
	default:
		//sprintf(buffer, "Reserved (%i)", i);
		//return buffer;
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Color Reserved format!");
	}
}
int gp_get_color_size(int i) {
	switch(i) {
	case 0:
		return 2;
	case 1:
		return 3;
	case 2:
		return 4;
	case 3:
		return 2;
	case 4:
		return 3;
	case 5:
		return 4;
	default:
		//return 0;
		DEGUB("Reserved (%i)\n", i);
		throw hardware_fatal_exception("GP Color Reserved format!");
	}
}

const char *gp_get_enc_string(int i) {
	switch(i) {
	case 0:
		return "not present";
	case 1:
		return "enc=direct";
	case 2:
		return "enc=i8";
	case 3:
		return "enc=i16";
	default:
		throw hardware_fatal_exception("Big Phat Error in gp_get_enc_string");
	}
}

DWORD gp_argb8_rgba4(WORD rgba4) {
	DWORD r = FOUR_TO_8BIT(getbitsr(rgba4, 15, 12));
	DWORD g = FOUR_TO_8BIT(getbitsr(rgba4, 11, 8));
	DWORD b = FOUR_TO_8BIT(getbitsr(rgba4, 7, 4));
	DWORD a = FOUR_TO_8BIT(getbitsr(rgba4, 3, 0));
	return (a << 24) | (r << 16) | (g << 8) | (b);
}
DWORD gp_argb8_rgba6(DWORD rgba6) {
	DWORD r = SIX_TO_8BIT(getbitsr(rgba6, 23, 18));
	DWORD g = SIX_TO_8BIT(getbitsr(rgba6, 17, 12));
	DWORD b = SIX_TO_8BIT(getbitsr(rgba6, 11, 6));
	DWORD a = SIX_TO_8BIT(getbitsr(rgba6, 5, 0));
	return (a << 24) | (r << 16) | (g << 8) | (b);
}
DWORD gp_argb8_rgb565(WORD rgb565) {
	DWORD r = FIVE_TO_8BIT(getbitsh(rgb565, 0, 4));
	DWORD g = SIX_TO_8BIT(getbitsh(rgb565, 5, 10));
	DWORD b = FIVE_TO_8BIT(getbitsh(rgb565, 11, 15));
	DWORD a = 0xFF;
	return (a << 24) | (r << 16) | (g << 8) | (b);
}
